// src/color.ts
import { getCSSVar } from "@chakra-ui/styled-system";
import {
  toHex,
  parseToRgba,
  transparentize as setTransparency,
  mix,
  darken as reduceLightness,
  lighten as increaseLightness,
  getContrast,
  parseToHsla,
  hsla,
  getLuminance
} from "color2k";

// ../../../node_modules/.pnpm/dlv@1.1.3/node_modules/dlv/dist/dlv.es.js
function dlv_es_default(t, e, l, n, r) {
  for (e = e.split ? e.split(".") : e, n = 0; n < e.length; n++)
    t = t ? t[e[n]] : r;
  return t === r ? l : t;
}

// src/color.ts
var isEmptyObject = (obj) => Object.keys(obj).length === 0;
var getColor = (theme, color, fallback) => {
  const hex = dlv_es_default(theme, `colors.${color}`, color);
  try {
    toHex(hex);
    return hex;
  } catch {
    return fallback != null ? fallback : "#000000";
  }
};
var getColorVar = (theme, color, fallback) => {
  var _a;
  return (_a = getCSSVar(theme, "colors", color)) != null ? _a : fallback;
};
var getBrightness = (color) => {
  const [r, g, b] = parseToRgba(color);
  return (r * 299 + g * 587 + b * 114) / 1e3;
};
var tone = (color) => (theme) => {
  const hex = getColor(theme, color);
  const brightness = getBrightness(hex);
  const isDark2 = brightness < 128;
  return isDark2 ? "dark" : "light";
};
var isDark = (color) => (theme) => tone(color)(theme) === "dark";
var isLight = (color) => (theme) => tone(color)(theme) === "light";
var transparentize = (color, opacity) => (theme) => {
  const raw = getColor(theme, color);
  return setTransparency(raw, 1 - opacity);
};
var whiten = (color, amount) => (theme) => {
  const raw = getColor(theme, color);
  return toHex(mix(raw, "#fff", amount));
};
var blacken = (color, amount) => (theme) => {
  const raw = getColor(theme, color);
  return toHex(mix(raw, "#000", amount / 100));
};
var darken = (color, amount) => (theme) => {
  const raw = getColor(theme, color);
  return toHex(reduceLightness(raw, amount / 100));
};
var lighten = (color, amount) => (theme) => {
  const raw = getColor(theme, color);
  toHex(increaseLightness(raw, amount / 100));
};
var contrast = (fg, bg) => (theme) => getContrast(getColor(theme, bg), getColor(theme, fg));
var isAccessible = (textColor, bgColor, options) => (theme) => isReadable(getColor(theme, bgColor), getColor(theme, textColor), options);
function isReadable(color1, color2, wcag2 = { level: "AA", size: "small" }) {
  var _a, _b;
  const readabilityLevel = readability(color1, color2);
  switch (((_a = wcag2.level) != null ? _a : "AA") + ((_b = wcag2.size) != null ? _b : "small")) {
    case "AAsmall":
    case "AAAlarge":
      return readabilityLevel >= 4.5;
    case "AAlarge":
      return readabilityLevel >= 3;
    case "AAAsmall":
      return readabilityLevel >= 7;
    default:
      return false;
  }
}
function readability(color1, color2) {
  return (Math.max(getLuminance(color1), getLuminance(color2)) + 0.05) / (Math.min(getLuminance(color1), getLuminance(color2)) + 0.05);
}
var complementary = (color) => (theme) => {
  const raw = getColor(theme, color);
  const hsl = parseToHsla(raw);
  const complementHsl = Object.assign(hsl, [
    (hsl[0] + 180) % 360
  ]);
  return toHex(hsla(...complementHsl));
};
function generateStripe(size = "1rem", color = "rgba(255, 255, 255, 0.15)") {
  return {
    backgroundImage: `linear-gradient(
    45deg,
    ${color} 25%,
    transparent 25%,
    transparent 50%,
    ${color} 50%,
    ${color} 75%,
    transparent 75%,
    transparent
  )`,
    backgroundSize: `${size} ${size}`
  };
}
var randomHex = () => `#${Math.floor(Math.random() * 16777215).toString(16).padEnd(6, "0")}`;
function randomColor(opts) {
  const fallback = randomHex();
  if (!opts || isEmptyObject(opts)) {
    return fallback;
  }
  if (opts.string && opts.colors) {
    return randomColorFromList(opts.string, opts.colors);
  }
  if (opts.string && !opts.colors) {
    return randomColorFromString(opts.string);
  }
  if (opts.colors && !opts.string) {
    return randomFromList(opts.colors);
  }
  return fallback;
}
function randomColorFromString(str) {
  let hash = 0;
  if (str.length === 0)
    return hash.toString();
  for (let i = 0; i < str.length; i += 1) {
    hash = str.charCodeAt(i) + ((hash << 5) - hash);
    hash = hash & hash;
  }
  let color = "#";
  for (let j = 0; j < 3; j += 1) {
    const value = hash >> j * 8 & 255;
    color += `00${value.toString(16)}`.substr(-2);
  }
  return color;
}
function randomColorFromList(str, list) {
  let index = 0;
  if (str.length === 0)
    return list[0];
  for (let i = 0; i < str.length; i += 1) {
    index = str.charCodeAt(i) + ((index << 5) - index);
    index = index & index;
  }
  index = (index % list.length + list.length) % list.length;
  return list[index];
}
function randomFromList(list) {
  return list[Math.floor(Math.random() * list.length)];
}

export {
  getColor,
  getColorVar,
  tone,
  isDark,
  isLight,
  transparentize,
  whiten,
  blacken,
  darken,
  lighten,
  contrast,
  isAccessible,
  isReadable,
  readability,
  complementary,
  generateStripe,
  randomColor
};
//# sourceMappingURL=chunk-6IC2I3BY.mjs.map