import { ThemingProps, StyleProps, SystemStyleObject } from '@chakra-ui/system';
import { ToastPosition } from './toast.placement.mjs';
import { AlertProps, AlertStatus } from '@chakra-ui/alert';
import * as react from 'react';

interface ToastProps extends UseToastOptions, Omit<AlertProps, keyof UseToastOptions> {
    onClose?: () => void;
}
/**
 * The `Toast` component is used to give feedback to users after an action has taken place.
 *
 * @see Docs https://chakra-ui.com/docs/components/toast
 */
declare const Toast: React.FC<ToastProps>;
declare function createRenderToast(options?: UseToastOptions & {
    toastComponent?: React.FC<ToastProps>;
}): react.FC<RenderProps>;
type UseToastPromiseOption = Omit<UseToastOptions, "status">;
declare function createToastFn(dir: "ltr" | "rtl", defaultOptions?: UseToastOptions): {
    (options?: UseToastOptions): ToastId;
    update(id: ToastId, options: Omit<UseToastOptions, "id">): void;
    promise<Result extends unknown, Err extends Error = Error>(promise: Promise<Result>, options: {
        success: MaybeFunction<UseToastPromiseOption, [Result]>;
        error: MaybeFunction<UseToastPromiseOption, [Err]>;
        loading: UseToastPromiseOption;
    }): void;
    closeAll: (options?: CloseAllToastsOptions | undefined) => void;
    close: (id: ToastId) => void;
    isActive: (id: ToastId) => boolean;
};
type CreateToastFnReturn = ReturnType<typeof createToastFn>;
type MaybeFunction<T, Args extends unknown[] = []> = T | ((...args: Args) => T);

interface UseToastOptions extends ThemingProps<"Alert"> {
    /**
     * The placement of the toast
     *
     * @default "bottom"
     */
    position?: ToastPosition;
    /**
     * The delay before the toast hides (in milliseconds)
     * If set to `null`, toast will never dismiss.
     *
     * @default 5000 ( = 5000ms )
     */
    duration?: ToastOptions["duration"];
    /**
     * Render a component toast component.
     * Any component passed will receive 2 props: `id` and `onClose`.
     */
    render?(props: RenderProps): React.ReactNode;
    /**
     * The title of the toast
     */
    title?: React.ReactNode;
    /**
     * The description of the toast
     */
    description?: React.ReactNode;
    /**
     * If `true`, toast will show a close button
     * @default false
     */
    isClosable?: boolean;
    /**
     * The status of the toast.
     */
    status?: AlertStatus;
    /**
     * A custom icon that will be displayed by the toast.
     */
    icon?: React.ReactNode;
    /**
     * The `id` of the toast.
     *
     * Mostly used when you need to prevent duplicate.
     * By default, we generate a unique `id` for each toast
     */
    id?: ToastId;
    /**
     * Callback function to run side effects after the toast has closed.
     */
    onCloseComplete?: () => void;
    /**
     * Optional style overrides for the container wrapping the toast component.
     */
    containerStyle?: StyleProps;
}
/**
 * React hook used to create a function that can be used
 * to show toasts in an application.
 */
declare function useToast(options?: UseToastOptions): CreateToastFnReturn;

interface RenderProps extends UseToastOptions {
    /**
     * Function to close the toast
     */
    onClose(): void;
}
type ToastMessage = (props: RenderProps) => React.ReactNode;
type ToastId = string | number;
interface ToastOptions {
    /**
     * The element or component type to render.
     * The component will be passed `id` and `onClose`
     */
    message: ToastMessage;
    /**
     * The toast's id
     */
    id: ToastId;
    /**
     * The duration of the toast
     */
    duration: number | null;
    /**
     * The status of the toast's alert component.
     */
    status: ToastStatus;
    /**
     * Function that removes the toast from manager's state.
     */
    onRequestRemove(): void;
    /**
     * The position of the toast
     */
    position: ToastPosition;
    /**
     * Callback function to run side effects after the toast has closed.
     */
    onCloseComplete?(): void;
    /**
     * Internally used to queue closing a toast. Should probably not be used by
     * anyone else, but documented regardless.
     */
    requestClose?: boolean;
    /**
     * Optional style overrides for the toast component.
     */
    containerStyle?: SystemStyleObject;
}
type ToastState = {
    [K in ToastPosition]: ToastOptions[];
};
type ToastStatus = "default" | "success" | "error" | "warning" | "info" | "loading";
type UpdateFn = (state: ToastState) => void;
type CloseAllToastsOptions = {
    positions?: ToastPosition[];
};

export { CloseAllToastsOptions as C, RenderProps as R, ToastState as T, UseToastOptions as U, ToastId as a, ToastOptions as b, ToastMessage as c, CreateToastFnReturn as d, Toast as e, createToastFn as f, createRenderToast as g, ToastProps as h, ToastStatus as i, UpdateFn as j, useToast as u };
